/** @file   ammunition.cpp
 * @brief   Implementation of Ammunition - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#include "ammunition.h"
#include "www_map.h"
#include "consts.h"
#include "WarGlobals.h"

using namespace eng2d;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
Ammunition::Ammunition() :
  iOwner( 0 ),
  iPosition( 0, 0 ),
  iAltitude( 0 ),
  iVerticalSpeed( 0 ),
  iVelocity( 0, 0 ),
  iFlags( 0 ),
  iDone( false )
{
}



/** Constructor
 */
Ammunition::Ammunition( GameObject* aOwner ) :
  iOwner( aOwner ),
  iPosition( 0,0 ),
  iAltitude( 0 ),
  iVerticalSpeed( 0 ),
  iVelocity( 0,0 ),
  iFlags( 0 ),
  iDone( false )
{
}



/** Destructor
 */
Ammunition::~Ammunition()
{
}




///
/// Public methods
/// ==============

/** Updates this ammunition
 */
bool Ammunition::update()
{
  bool c = false;
  // A quick hack that allowes the ammunition to move faster. Normally if the
  // bullet goes faster than twice the object's bounding sphere, it's not
  // possible to detect every bullet hits to that object.
  if ( iVelocity.length2() > 400 ) {
    Vec2D mvec = iVelocity;
    mvec *= 0.5;
    iPosition += mvec;
    c = Map::bulletCollide(iPosition);
    if ( c != true ) {
      iPosition += mvec;
      c = Map::bulletCollide(iPosition);
    }
  } else {
    iPosition += iVelocity;
    c = Map::bulletCollide(iPosition);
  }
  if ( c == true ) {
    iDone = true;
    // The bullet did collide. Play sound and bounce
    if ( this->hasFlag( KMakeSound ) == true ) {
      Vec2D sparksDir = Vec2D() - iVelocity;
      ParticleSystem* particles = new ParticleGunFlames( iPosition, sparksDir, 15, 60 );
      WarGlobals::pPartManager->addSystem( particles );
    }

    if ( this->hasFlag( KBounce ) == true ) {
      this->bounce();
      c = false;
      iDone = false;
    }
  }

  // Don't let the ammunition exceed the map boundaries.
  if ( iPosition.vx < 0 || iPosition.vy < 0 ||
    iPosition.vx > Map::mwidth || iPosition.vy > Map::mheight ) {
    // Mark this ammunition as ready to be deleted.
    iDone = true;
  }
  
  // Modify the altitude
  iAltitude += iVerticalSpeed;
  iVerticalSpeed -= Consts::GRAVITY;
  
  // Update the animation
  iAnimation.update();
  if ( iAnimation.paused() == true ) {
    // The animation has been played through so we can delete the bullet.
    iDone = true;
  }
  return this->updateAmmunition();
}



/** Sets the velocity
 */
void Ammunition::setVelocity( const Vec2D& aVelocity )
{
  iVelocity = aVelocity;
  iVelocityVectorLength = iVelocity.length();
}



/** Ending actions
 */
void Ammunition::endingActions( BulletTable* aBulletTable )
{
}



///
/// Getter methods
/// ==============

/** Tells if at least one of the given flags is on.
 */
bool Ammunition::hasFlag( TFlags aFlagMask ) const
{
  return static_cast<bool>( iFlags & aFlagMask );
}




///
/// Private or Protected methods
/// ============================

/** Makes this ammunition to bounce
 */
void Ammunition::bounce()
{
  // Since this method gets called when the collision has already taken place.
  // First we undo the movement
  iPosition -= iVelocity;

  // Next try the movement only by the x and y components.
  // That way we find out wich direction the bullet can bounce
  Vec2D xmove( iPosition.vx + iVelocity.vx, iPosition.vy );
  Vec2D ymove( iPosition.vx, iPosition.vy + iVelocity.vy );
  bool xhit = Map::bulletCollide( xmove );
  bool yhit = Map::bulletCollide( ymove );
  if ( xhit == true && yhit == false ) {
    // Bounce horizontally
    iVelocity.vx = -iVelocity.vx;

  } else if ( xhit == false && yhit == true ) {
    // Bounce vertically
    iVelocity.vy = -iVelocity.vy;

  } else if ( xhit == true && yhit == true ) {
    // Bounce back
    iVelocity.vx = -iVelocity.vx;
    iVelocity.vy = -iVelocity.vy;

  }
}

} // end of namespace
